/**
* \file: Server.cpp
*
* \version: $Id:$
*
* \release: $Name:$
*
* <brief description>.
* <detailed description>
* \component: Baidu CarLife
*
* \author: P. Govindaraju / RBEB/GM / Pradeepa.Govindaraju@in.bosch.com
*
* \copyright (c) 2016 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
* \see <related items>
*
* \history
*
***********************************************************************/

#include <adit_logging.h>
#include "Server.h"
#include <aoap_types.h>
#include <iostream>
#include <fstream>
#include <aoap.h>
#include <sstream>

LOG_IMPORT_CONTEXT(tbdcl)

#define TIMEOUT_PROTOCOL 8
#define ACCESSORY_MANUFACTURER_NAME     "Baidu"
#define ACCESSORY_MODEL_NAME            "CarLife"
#define ACCESSORY_DESCRIPTION           "Baidu CarLife"
#define ACCESSORY_VERSION               "1.0.0"
#define ACCESSORY_URI                   "http://carlife.baidu.com"
#define ACCESSORY_SERIAL_NUMBER         "0720SerialNo"

namespace adit { namespace bdcl {

S_CAR_GPS Server::myGpsData = {1,0,0,0,0,0,0,2016,4,14,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
S_MOBILE_CARLIFE_INFO Server::carlifeInfo;
S_SUBSCRIBE_MOBILE_CARLIFE_INFO_LIST Server::carlifeInfoList;
S_VEHICLE_INFO Server::carInfo;
S_VEHICLE_INFO_LIST Server::carInfoList;
S_FEATURE_CONFIG_LIST Server::configList;
using namespace std;
using namespace utility;
using namespace uspi;

Server::Server() : coreChannel(nullptr), videoStream(nullptr), medAudio(nullptr), vrPlayback(nullptr), ttsAudio(nullptr),
        vrCapture(nullptr), touchInput(nullptr)
{
    sessionActive = false;

    running = false;

    is_teardown = false;
    thread_running = false;
    keyevent_thread = -1;

    mSwitchResult = 0;
    layerManager = nullptr;

    miAP2EASessionRunning = false;
    miAP2Session = nullptr;
    miAP2DeviceSerial = "";
    startEventDispatcher();
    isDestroySession = false;
    sendVideoEncoderConditionalFlag = false;
}

Server::~Server(void)
{
    stopEventDispatcher();
}

bool Server::init(const std::string& inConfigPath,const std::string& inChannelNumber)
{

    mCarlifeStaticsInfo.channel = inChannelNumber.c_str();
    setFeatureConfig();
// Set the flag in bdcl_seettings.mk to get the aoap verbose level logs
#ifdef AOAP_VERBOSE
    aoap_set_log_level(0x06, false);
#endif

    /* create bdcl-demo layer manager */
    layerManager = new bdclLayerManager();

    running = true;
    thread_running = false;
    return true;
}

bool Server::cleanUp()
{
    if (layerManager != nullptr)
    {
        LOG_INFO((tbdcl, "%s()  Delete layerManager", __FUNCTION__));
        delete layerManager;
        layerManager = nullptr;
    }
    else
    {
        LOG_INFO((tbdcl, "%s()  layerManager already deleted", __FUNCTION__));
    }

    return true;
}

bool Server::layerManagerInit()
{
    if (layerManager == nullptr)
    {
        LOG_ERROR((tbdcl, "could not allocate resource of bdclLayerManager"));
        return false;
    }
    else
    {
        if(!layerManager->init())
        {
            LOG_ERROR((tbdcl, "layerManager Init has failed..."));
            return false;
        }
    }
    return true;
}

bool Server::switchDevice(uint32_t inVid, uint32_t inPid, std::string inSerial,
                          aoapTransportInfo_t* outAoapTransportInfo)
{
    const unsigned int timeout = 10000;

    // fill AccessoryInfo to identify & start the baidu application on the MD
    aoapDeviceInfo_t aoapDeviceInfo;
    aoapDeviceInfo.aoapAccessoryInfo.manufacturer = ACCESSORY_MANUFACTURER_NAME;
    aoapDeviceInfo.aoapAccessoryInfo.modelName = ACCESSORY_MODEL_NAME;
    aoapDeviceInfo.aoapAccessoryInfo.description = ACCESSORY_DESCRIPTION;
    aoapDeviceInfo.aoapAccessoryInfo.version = ACCESSORY_VERSION;
    aoapDeviceInfo.aoapAccessoryInfo.uri = ACCESSORY_URI;
    aoapDeviceInfo.aoapAccessoryInfo.serial = ACCESSORY_SERIAL_NUMBER;
    /* Enable Audio support.
     * AOA 2.0 includes optional the support for audio output from the MD to the HU.
     * AOA 1.0 does not supports the optional audio output.
     * Because bdcl supports AOA 1.0, set enableAudio to zero. */
    aoapDeviceInfo.aoapAccessoryInfo.enableAudio = 0;

    // fill aoapDeviceInfo_t structure to switch the MD into Accessory mode
    aoapDeviceInfo.vendorId = inVid;
    aoapDeviceInfo.productId = inPid;
    // TODO who does free this?
    aoapDeviceInfo.pSerial = strdup(inSerial.c_str());

    AoapDevice aoapDevice;
    /* switch the MD into Accessory mode.
     * this will not start the communication with between the MD and the GalReceiver. */
    int result = aoapDevice.switchDevice(&aoapDeviceInfo, outAoapTransportInfo, timeout);
    if (0 == result) {
        LOG_INFO((tbdcl, "Start Baidu on device (vendorId=%X, producId=%X, serial=%s)",
                aoapDeviceInfo.vendorId, aoapDeviceInfo.productId, aoapDeviceInfo.pSerial));
        return true;
    }
    else {
        LOG_ERROR((tbdcl, "Failed to start Baidu on device (vendorId=%X, producId=%X, " \
                "serial=%s) = %d",
                aoapDeviceInfo.vendorId, aoapDeviceInfo.productId, aoapDeviceInfo.pSerial,
                result));
        return false;
    }
}

bool Server::createEndpoints()
{
    mCallbackDealer = new CoreCallbackDealer();
    coreChannel = new AditCoreSurrogateChannel(mCallbackDealer);
    if (!coreChannel)
    {
        LOG_ERROR((tbdcl,"could not create core surrogate channel"));
        return false;
    }

    touchInput = new InputChannel(mCallbackDealer);
    if(!touchInput)
    {
        LOG_ERROR((tbdcl,"could not create inputsource"));
        return false;
    }

    videoStream = new VideoChannel(mCallbackDealer);
    if (!videoStream)
    {
        LOG_ERROR((tbdcl,"could not create Video channel"));
        return false;
    }

    medAudio = new MediaAudioChannel(mCallbackDealer);
    if (!medAudio)
    {
        LOG_ERROR((tbdcl,"could not create mediaaudio"));
        return false;
    }

    ttsAudio = new NaviTtsAudioChannel(mCallbackDealer);
    if (!ttsAudio)
    {
        LOG_ERROR((tbdcl,"could not create ttsAudio"));
        return false;
    }

    vrPlayback = new VrTtsAudioChannel(mCallbackDealer);
    if (!vrPlayback)
    {
        LOG_ERROR((tbdcl,"could not create vrPlayback"));
        return false;
    }

    vrCapture = new VrCaptureAudioChannel(mCallbackDealer);
    if (!vrCapture)
    {
        LOG_ERROR((tbdcl,"could not create vrCapture"));
        return false;
    }

    return true;
}

#define STDIN 0

void Server::removeDot(string& inStr, int len)
{
    int j = 0;
    for(int i = 0; i < len; i++)
    {
        if(inStr[i] == '.')
        {
            continue;
        }
        else
        {
            inStr[j] = inStr[i];
            j++;
        }
    }
    inStr[j] = '\0';
}
void *Server::keyeventListener(void *arg)
{

    while(!Server::instance().is_teardown)
    {
        S_MODULE_STATUS_CONTROL status;
        fd_set fds;
        struct timeval tv;
        FD_ZERO(&fds);
        FD_SET(STDIN, &fds);
        tv.tv_sec = 0;
        tv.tv_usec = 50 * 1000; // tbd
        std::condition_variable cvGpsTimeOut;
        std::mutex mutGpsTimeOut;
        int ret = select(STDIN+1,&fds,NULL,NULL,&tv);
        if (ret == 1)
        {
            int key = getchar();
            switch (key)
            {
            case 'A':
                printf("\n\nSetting media to idle\n\n");

                status.moduleID = MEDIA_CHANNEL;
                status.statusID = 0; // MUSIC_STATUS_IDLE
                CCarLifeLib::getInstance()->cmdModuleControl(&status);
                break;
            case 'S':
                printf("\n\nSetting media to running\n\n");
                status.moduleID = MEDIA_CHANNEL;
                status.statusID = 1; // MUSIC_STATUS_RUNNING
                CCarLifeLib::getInstance()->cmdModuleControl(&status);
                break;
            case 'Q':
                printf("\n\nSetting video to idle\n\n");

                status.moduleID = VIDEO_CHANNEL;
                status.statusID = 0; // MUSIC_STATUS_IDLE
                CCarLifeLib::getInstance()->cmdModuleControl(&status);
                break;
            case 'W':
                printf("\n\nSetting video to running\n\n");
                status.moduleID = VIDEO_CHANNEL;
                status.statusID = 1; // MUSIC_STATUS_RUNNING
                CCarLifeLib::getInstance()->cmdModuleControl(&status);
                break;
            case 'R':
                printf("\n\nSetting start video encoder\n\n");
                CCarLifeLib::getInstance()->cmdVideoEncoderStart();
                break;
            case 'T':
                printf("\n\nSetting pause video encoder\n\n");
                CCarLifeLib::getInstance()->cmdVideoEncoderPause();
                break;
            case 'V':
                printf("\n\nSetting video encoder to Reset\n\n");
                CCarLifeLib::getInstance()->cmdVideoEncoderReset();
                break;
            case 'G':
            {
                cout << "sending GPS data" << endl;
                std::cv_status gpsTimeOutStatus;

                ifstream gpsData ("/opt/platform/gpstest_1.txt");
                if(gpsData.is_open())
                {
                    std::string line;
                    std::unique_lock<std::mutex> lckProtocolVersion(mutGpsTimeOut);
                    while ( getline (gpsData,line) )
                    {
                        std::string delimiter = ",";
                        unsigned int pos = 0;
                        int index = 0;
                        std::string token1;
                        std::string token2;
                        std::string tmp;
                        std::string c = ".";
                        while(((pos = line.find(delimiter)) != std::string::npos) && index < 3)
                        {
                            tmp = line.substr(0, pos);

                            if(index > 0)
                            {
                                if(index == 1)
                                {
                                    token1 = tmp;
                                    Server::instance().removeDot(token1, token1.length());
                                }
                                else
                                {
                                    token2 = tmp;
                                    Server::instance().removeDot(token2, token2.length());
                                }
                            }
                            line.erase(0, pos + delimiter.length());
                            index++;
                        }
                        myGpsData.latitude = atoi(token2.c_str());
                        myGpsData.longitude = atoi(token1.c_str());
                        myGpsData.speed = 1000;
                        myGpsData.satsUsed = 20;
                        myGpsData.satsVisible = 10;
                        cout << "latitude=" << myGpsData.latitude << " longitude= " << myGpsData.longitude << endl;
                        CCarLifeLib::getInstance()->cmdCarGPS(&myGpsData);
                        gpsTimeOutStatus = cvGpsTimeOut.wait_for(lckProtocolVersion,std::chrono::seconds(1));
                        if(gpsTimeOutStatus == std::cv_status::no_timeout)
                        {
                            cout << "breaking the loop" << endl;
                            break;
                        }

                    }
                    gpsData.close();
                }
                else
                {
                    cout << "unable to open the file" << endl;
                }
                break;
            }
            case 10:
                break; // backspace
            default:
                printf("pressed unkown key: %c (%d)\n", (char) key, key);
                break;
            }
        }
    }

    return NULL;

}

bool Server::startSession(aoapTransportInfo_t inAoapTransportInfo)
{
    bdclInitStatus startSessionStatus = BDCLInit_NoError;
    bool rc;
    rc = Server::instance().configBdcl();
    if(!rc)
    {
        //error is already logged.
        return false;
    }

    startSessionStatus = coreChannel->mCoreSurrogate->initialize(inAoapTransportInfo.aoapAccessoryId,
            inAoapTransportInfo.aoapDeviceId);
    if (startSessionStatus == BDCLInit_HUVersionError || startSessionStatus == BDCLInit_StatisticInfoError ||
            startSessionStatus == BDCLInit_ParamError)
    {
        LOG_INFO((tbdcl, "bdclInit is Failed"));
        destroySession();
        return false;
    }
    else if(startSessionStatus == BDCLInit_NoError)
    {
        LOG_INFO((tbdcl, "bdclInit is successfull"));
        sessionActive = true;

    }

    if (!sendVideoEncoderConditionalFlag)
    {
        std::unique_lock < std::mutex > lckSendVideoEncoder(mutSendVideoEncoder);
        cvSendVideoEncoder.wait(lckSendVideoEncoder);
    }

    if(!isDestroySession)
    {
        if( videoStream )
        {
            videoStream->configureCodec();
        }

        is_teardown=false;
        if (AutoSmoketest::instance().getTestMode() == MANUAL)
        {Server::instance().configBdcl();
            if (!pthread_create(&keyevent_thread, NULL, keyeventListener, NULL))
            {
                thread_running = true;
            }

            int rc = 0;
            rc = pthread_setname_np(keyevent_thread, "keyevent");
            if (rc != 0)
            {
                LOG_WARN((tbdcl,"error in setting thread Name for keyevent\n"));
            }
        }

        CCarLifeLib::getInstance()->cmdHUInfro(&(Server::instance().huInfo));
        CCarLifeLib::getInstance()->cmdCarLifeDataSubscribe(&carlifeInfoList);
    }
    return true;

}

bool Server::startSession(const char* wFile, const char* rFile)
{
    bdclInitStatus startSessionStatus = BDCLInit_NoError;
    bool rc;
    rc = Server::instance().configBdcl();
    if(!rc)
    {
        //error is already logged.
        return false;
    }

    startSessionStatus = coreChannel->mCoreSurrogate->initialize(wFile, rFile);
    if (startSessionStatus == BDCLInit_HUVersionError || startSessionStatus == BDCLInit_StatisticInfoError ||
            startSessionStatus == BDCLInit_ParamError)
    {
        LOG_INFO((tbdcl, "bdclInit is for iOS  Failed"));
        destroySession();
        return false;
    }
    else if(startSessionStatus == BDCLInit_NoError)
    {
        LOG_INFO((tbdcl, "bdclInit for iOS is successfull"));
        sessionActive = true;

    }

    if (!sendVideoEncoderConditionalFlag)
    {
        std::unique_lock < std::mutex > lckSendVideoEncoder(mutSendVideoEncoder);
        cvSendVideoEncoder.wait(lckSendVideoEncoder);
    }

    if (!isDestroySession )
    {
        if( videoStream )
        {
            videoStream->configureCodec();
        }

        is_teardown=false;
        if (AutoSmoketest::instance().getTestMode() == MANUAL)
        {
            if (!pthread_create(&keyevent_thread, NULL, keyeventListener, NULL))
            {
                thread_running = true;
            }

            int rc = 0;
            rc = pthread_setname_np(keyevent_thread, "keyevent");
            if (rc != 0)
            {
                LOG_WARN((tbdcl,"error in setting thread Name for keyevent\n"));
            }
        }

        CCarLifeLib::getInstance()->cmdHUInfro(&(Server::instance().huInfo));
        CCarLifeLib::getInstance()->cmdCarLifeDataSubscribe(&carlifeInfoList);

    }
    return true;

}

bool Server::configBdcl()
{

    sendVideoEncoderConditionalFlag = false;
    isDestroySession = false;
    AutoSmoketest::instance().setTestError(GSTINTERNAL);

    if ( !layerManagerInit() ) {
        //error is already logged.
        return false;
    }
    if (!createEndpoints()) {
        LOG_INFO((tbdcl, "Failed to create endpoints"));
        return false;
    }
    //set the timout for protocol match status message
    coreChannel->mCoreSurrogate->setProtocolStatusTimeout(TIMEOUT_PROTOCOL);
    // set the Staticstic Info before callilng bdclInit
    coreChannel->mCoreSurrogate->setCarLifeStaticsInfo(&mCarlifeStaticsInfo);
    //CarlifeInfo
    carlifeInfo.moduleID = CARLIFE_DATA_TURNBYTURN;
    carlifeInfo.pNext = NULL;
    carlifeInfoList.cnt = 1;
    carlifeInfoList.pMobileCarLifeInfo = &carlifeInfo;
    //Car Info
    carInfo.moduleID = CAR_DATA_GPS;
    carInfo.supportFlag = 1;
    carInfo.pNext = NULL;
    carInfoList.cnt = 1;
    carInfoList.pVehicleInfo = &carInfo;

    //0x00010044
    CCarLifeLib::getInstance()->cmdRegisterCarLifeDataSubscribeDone(cmdCarlifeDataSubscribeDone);
    //0x00010030
    CCarLifeLib::getInstance()->cmdRegisterNaviNextTurnInfo(cmdNaviNextTurnInfo);
    //0x00050005
    CCarLifeLib::getInstance()->vrRegisterStatusModule(vrStatusModule);
    //0x00010059
    CCarLifeLib::getInstance()->cmdRegisterMdExit(cmdMdExit);
    //0x00010031
    CCarLifeLib::getInstance()->cmdRegisterCarDataSubscribe(cmdCarDataSubscribe);
    //0x00010033
    CCarLifeLib::getInstance()->cmdRegisterCarDataSubscribeStart(cmdCarDataSubscribeStart);
    //0x00010034
    CCarLifeLib::getInstance()->cmdRegisterCarDataSubscribeStop(cmdCarDataSubscribeStop);
    //0x00010051
    CCarLifeLib::getInstance()->cmdRegisterFeatureConfigRequest(cmdRegisterFeatureConfigRequest);
    return true;
}

void Server::cmdCarlifeDataSubscribeDone(S_SUBSCRIBE_MOBILE_CARLIFE_INFO_LIST* list)
{
    //0x00018045
    CCarLifeLib::getInstance()->cmdCarLifeDataSubscribeStart(&carlifeInfoList);
    LOG_INFO((tbdcl, "%s ", __FUNCTION__));
    if((list->pMobileCarLifeInfo->supportFlag) == true)
    {
        LOG_INFO((tbdcl, "count = %d, moduleID = %d", list->cnt, list->pMobileCarLifeInfo->moduleID));
    }
    else
    {
        LOG_INFO((tbdcl, "Subscribed module not supported"));
    }

}

void Server::cmdNaviNextTurnInfo(S_NAVI_NEXT_TURN_INFO* naviInfo)
{
    LOGD_DEBUG((tbdcl, "%s action = %d, nextTurn = %d, roadName = %s, totalDistance = %d, remainDistance = %d, turnIconData = %s",
            __FUNCTION__, naviInfo->action, naviInfo->nextTurn, naviInfo->roadName.c_str(), naviInfo->totalDistance, naviInfo->remainDistance,
            naviInfo->turnIconData.c_str()));
}
void Server::vrStatusModule (S_VR_STATUS_LIST* vrStatusId)
{
    LOGD_DEBUG((tbdcl, "VR Status Module is invoked count = %d", vrStatusId->cnt));
    if(vrStatusId->cnt > 0)
    {
    LOGD_DEBUG((tbdcl, "VR Status Module is invoked moduelID = %d and statusID = %d", vrStatusId->moduleStatus->moduleID,
            vrStatusId->moduleStatus->statusID));
    }
}
// cmdMdExit callback triggered by bdclCore.
void Server::cmdMdExit()
{
    LOGD_DEBUG((tbdcl, "%s user exited the application", __FUNCTION__));
    std::shared_ptr<EventItem> event(new EventItem(nullptr, EventType::CMDMDEXIT,"",""));
    Server::instance().queueEvent(event);
}

void Server::onMDExit(std::string inString)
{
    LOG_INFO((tbdcl, "%s triggered with %s", __FUNCTION__, inString.c_str()));
    destroySession();
}

void Server::onAoapReadTimeOutError(std::string inString)
{
    LOG_INFO((tbdcl, "%s triggered with %s", __FUNCTION__, inString.c_str()));
    destroySession();
}

void Server::destroySession(void)
{
    std::unique_lock<std::mutex> lckBdclDestroy(mutBdclDestroy);
    isDestroySession = true;
    Server::instance().sendVideoEncoder(true);
    if(thread_running)
    {
        is_teardown=true;
        pthread_join(keyevent_thread,NULL);
        thread_running=false;
    }

    if (sessionActive)
    {
        LOG_INFO((tbdcl," %s teardown :sessionActive is false", __FUNCTION__));

        if (vrCapture != nullptr)
        {
            LOG_INFO((tbdcl,"%s()  Tear down VR channel", __FUNCTION__));
            vrCapture->teardown();
        }

        coreChannel->mCoreSurrogate->teardown();
        sessionActive = false;
    }
    else
    {
        LOG_INFO((tbdcl, "%s()  Session not active (sessionActive: %d)", __FUNCTION__, sessionActive));
    }

    if ( coreChannel )
    {
        delete coreChannel;
        coreChannel = nullptr;
    }

    if ( medAudio )
    {
        delete medAudio;
        medAudio = nullptr;
    }

    if ( vrPlayback )
    {
        delete vrPlayback;
        vrPlayback = nullptr;
    }

    if ( ttsAudio )
    {
        delete ttsAudio;
        ttsAudio = nullptr;
    }

    if ( videoStream )
    {
        delete videoStream;
        videoStream = nullptr;
    }

    if( touchInput )
    {
        delete touchInput;
        touchInput=nullptr;
    }

    if ( mCallbackDealer )
    {
        delete mCallbackDealer;
        mCallbackDealer = nullptr;
    }

    CCarLifeLib *pCCarLifeLib=CCarLifeLib::getInstance();
    pCCarLifeLib->carLifeLibDestory();
    if (layerManager != nullptr)
    {
        layerManager->finalize();
    }
}

void Server::requestStop()
{
    if((Server::instance().sessionActive))
    {
        LOG_INFO((tbdcl, "%s()  destroy Baidu session", __func__));
        Server::instance().destroySession();
    }
}

void Server::authenticatonResult(void)
{
    LOG_ERROR((tbdcl, "inside the authenticationResult"));

}

void Server::setFeatureConfig()
{
    configList.cnt=1;
    configList.pFeatureConfig=&featureConfigFocusUI;
    featureConfigFocusUI.key="MEDIA_SAMPLE_RATE";
    featureConfigFocusUI.value=1; //Forcing MD to send Media audio in 48kHz samplerate to reduce the high CPU load consumed by ALSA thread.
    featureConfigFocusUI.pNext=NULL;

}

void Server::playbackStopCallback()
{
    LOG_INFO((tbdcl, "playbackStopCallback called video pipeline is closed"));
}


void Server::sendVideoEncoder(bool inStatus)
{
    std::unique_lock < std::mutex > lckSendVideoEncoder(mutSendVideoEncoder);
    sendVideoEncoderConditionalFlag = inStatus;
    cvSendVideoEncoder.notify_one();
}

void Server::cmdCarDataSubscribe(S_VEHICLE_INFO_LIST*)
{
    LOGD_DEBUG((tbdcl,"cmdCarDataSubscribe is invoked"));
    CCarLifeLib::getInstance()->cmdCarDataSubscribeDone(&carInfoList);
    Server::instance().sendVideoEncoder(true);

}

void Server::cmdCarDataSubscribeStart(S_VEHICLE_INFO_LIST* inCarInfo)
{
    LOGD_DEBUG((tbdcl,"cmdCarDataSubscribeStart is invoked count= %d",inCarInfo->cnt));
    VehicleInfo* localVar =  inCarInfo->pVehicleInfo;
    for( unsigned int i = 0; i < inCarInfo->cnt; i++)
    {
        LOG_INFO((tbdcl,"cmdCarDataSubscribeStart moduelID = %d, supportFlag = %d",localVar->moduleID, localVar->supportFlag));
        localVar = localVar->pNext;
    }
}

void Server::cmdCarDataSubscribeStop(S_VEHICLE_INFO_LIST*)
{
    LOGD_DEBUG((tbdcl,"cmdCarDataSubscribeStop is invoked"));
}

//0x00010051
void Server::cmdRegisterFeatureConfigRequest()
{
    LOGD_DEBUG((tbdcl,"cmdRegisterFeatureConfigRequest is invoked"));
    //0x00018052
    CCarLifeLib::getInstance()->cmdFeatureConfigResponse(&configList);
}


void Server::onAppleDeviceFound(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice)
{
    SpiFeatureProtocol devProtocol = FD_PROTOCOL_APPLE_NATIVE_HOST_MODE;

    LOG_INFO((tbdcl, "Apple device found"));

    DeviceInfo ddInfo = inUsbDevice->getInfo();

    LOG_INFO((tbdcl, "%s()  Switch poduct=%s to Native Host Mode",
            __func__, ddInfo.getDeviceInfo(DSYS_PRODUCT).c_str() ));

    /* switch done by the FeatureDiscovery */
    if (DiscoveryError::OK != inUsbDevice->switchDevice(devProtocol, nullptr, 1000))
    {
        LOG_ERROR((tbdcl, "switchDevice() failed."));
    }
    else
    {
        LOG_INFO((tbdcl, "switchDevice() success."));

        /* remove if already stored */
        auto found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));
        if (found != transportInfos.end()) {
            transportInfos.erase(found);
        }

        /* insert empty information to indicate that the switch was triggered by us */
        aoapTransportInfo_t info = {0,0};
        transportInfos.insert(std::pair<string, aoapTransportInfo_t>(ddInfo.getDeviceInfo(DSYS_SERIAL), info));
    }
}

void Server::onAppleDeviceLost(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice)
{
    DeviceInfo ddInfo = inUsbDevice->getInfo();

    LOG_INFO((tbdcl, "Apple device lost"));

    // any lost device
    if((Server::instance().sessionActive))
    {
        LOG_INFO((tbdcl, "%s()  destroy baidu session %s", __func__, ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));
        // lost device is not necessarily one with a session
        Server::instance().destroySession();
    }
    else
    {

        /* prepare and initialize EA session */
        if (true == initEaSession(ddInfo.getDeviceInfo(DSYS_SERIAL))) {
            LOG_INFO((tbdcl, "%s()  iAP2 EA Session initializied successfully", __func__));
        } else {
            LOG_ERROR((tbdcl, "%s()  Init iAP2 EA Session failed", __func__));
        }

        /* switch the USB OTG port to gadget */
        if (true == startEaSession()) {
            LOG_INFO((tbdcl, "%s()  iAP2 EA Session started successfully", __func__));
        } else {
            LOG_ERROR((tbdcl, "%s()  Start iAP2 EA Session failed", __func__));
        }
    }
}

void Server::onAoapDeviceFound(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice)
{
    SpiFeatureProtocol devProtocol = FD_PROTOCOL_GOOGLE_AOAP;

    // any supported device
    DeviceInfo ddInfo = inUsbDevice->getInfo();

    /* check for AOAP support */
    if (true == inUsbDevice->checkDevice(devProtocol, 1000)) {
        LOG_INFO((tbdcl, "AOAP device found: poduct=%s",
                ddInfo.getDeviceInfo(DSYS_PRODUCT).c_str() ));

        // switch done by the FeatureDiscovery
        struct SpiAoapInformation tmpParam;
        tmpParam.manufacturer   = "Baidu";
        tmpParam.modelName      = "CarLife";
        tmpParam.description    = "Baidu CarLife";
        tmpParam.version        = "1.0.0";
        tmpParam.uri            = "http://carlife.baidu.com";
        tmpParam.serial         = "0720SerialNo";
        tmpParam.enableAudio    = 0;

        if (DiscoveryError::OK != inUsbDevice->switchDevice(FD_PROTOCOL_GOOGLE_AOAP, &tmpParam, 1000))
        {
            LOG_ERROR((tbdcl, "%s()  switchDevice() failed", __func__));
        }
        else
        {
            LOG_INFO((tbdcl, "%s()  switchDevice() success", __func__));

            // remove if already stored/
            auto found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));
            if (found != transportInfos.end()) {
                transportInfos.erase(found);
            }

            // insert empty information to indicate that the switch was triggered by us
            aoapTransportInfo_t info = {0,0};
            transportInfos.insert(std::pair<string, aoapTransportInfo_t>(ddInfo.getDeviceInfo(DSYS_SERIAL), info));

            AutoSmoketest::instance().setTestError(NOERROR);
        }
    } else {
        LOG_INFO((tbdcl, "%s()  device does not support AOAP", __func__));
        AutoSmoketest::instance().setTestError(AOAPNOTSUPPORTED);
    }
}

void Server::onAoapDeviceLost(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice)
{
    DeviceInfo ddInfo = inUsbDevice->getInfo();

    // any lost device
    LOG_INFO((tbdcl, "AOAP device lost"));

    // any lost device
    if((Server::instance().sessionActive))
    {
        LOG_INFO((tbdcl, "%s()  destroy baidu session %s", __func__, ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));
        // lost device is not necessarily one with a session
        Server::instance().destroySession();
    }
}

void Server::onDeviceSwitched(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice)
{
    pthread_mutex_lock(&Server::instance().sessionMutex);   // session mutex lock

    DeviceInfo ddInfo = inUsbDevice->getInfo();

    // any device in accessory mode, either switched by us or already in accessory mode
    LOG_INFO((tbdcl, "AOAP device found in AOAP mode product=%s %s",
            ddInfo.getDeviceInfo(DSYS_PRODUCT).c_str(), ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));

    auto found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));

    // check if transport info already exist for this device
    if (found == transportInfos.end())
    {
        // not found
        LOG_WARN((tbdcl, "%s()  accessory mode in device %s %s without previous switch connected",
                __func__, ddInfo.getDeviceInfo(DSYS_PRODUCT).c_str(), ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));

        // switch device and retrieve info
        aoapTransportInfo_t info = {0,0};
        bool switchResult = Server::instance().switchDevice(ddInfo.getiDeviceInfo(DSYS_IDVENDOR), \
                                                            ddInfo.getiDeviceInfo(DSYS_IDPRODUCT), \
                                                            ddInfo.getDeviceInfo(DSYS_SERIAL), &info);
        if (true != switchResult)
        {
            // TODO error handling
            LOG_ERROR((tbdcl, "%s()  switchDevice() failed", __func__));
        }
        else
        {
            LOG_INFO((tbdcl, "%s()  switchDevice() success", __func__));

            // remove if already stored
            // TODO do differently!
            auto found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));
            if (found != transportInfos.end())
            {
                transportInfos.erase(found);
            }

            transportInfos.insert(std::pair<string, aoapTransportInfo_t>(ddInfo.getDeviceInfo(DSYS_SERIAL), info));

            LOG_INFO((tbdcl, "%s()  re-create Baidu session %s", __func__, ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));

            Server::instance().startSession(info);
            // info is not guaranteed to live longer
            // TODO error handling

            // transport info no longer required
            found = transportInfos.find(ddInfo.getDeviceInfo(DSYS_SERIAL));
            if (found != transportInfos.end())
            {
                transportInfos.erase(found);
            }
        }
    }
    else
    {
        // transport info already exist

        // switch device to retrieve AOAP transport information
        aoapTransportInfo_t info = {0,0};
        bool switchResult = Server::instance().switchDevice(ddInfo.getiDeviceInfo(DSYS_IDVENDOR), \
                                                            ddInfo.getiDeviceInfo(DSYS_IDPRODUCT), \
                                                            ddInfo.getDeviceInfo(DSYS_SERIAL), &info);
        if (true != switchResult)
        {
            // TODO error handling
            LOG_ERROR((tbdcl, "%s()  switchDevice() failed", __func__));
        }
        else
        {
            LOG_INFO((tbdcl, "%s()  switchDevice() success", __func__));

            LOG_INFO((tbdcl, "%s()  create Baidu session %s, TransportInfo %d:%d",
                    __func__, to_string_server(ddInfo).c_str(), info.aoapAccessoryId, info.aoapDeviceId));
            Server::instance().startSession(info);
        }
        // transport info no longer required
        transportInfos.erase(found);
    }

    pthread_mutex_unlock(&Server::instance().sessionMutex);   // session mutex unlock
}

void Server::onDeviceChanged(std::shared_ptr<adit::uspi::DiscoveredDeviceUsb> inUsbDevice)
{
    DeviceInfo ddInfo = inUsbDevice->getInfo();
    LOG_INFO((tbdcl, "device %s changed", ddInfo.getDeviceInfo(DSYS_SERIAL).c_str()));
}

void Server::onDeviceError(adit::uspi::DiscoveryError inErrorCode)
{
    LOG_INFO((tbdcl, "errorCb triggered with error %s", to_str(inErrorCode)));
}

void Server::onStartEANative(std::string wFile, std::string rFile)
{

    LOG_INFO((tbdcl, "Server::%s()  wFile:%s, rFile:%s", __func__, wFile.c_str(), rFile.c_str()));
    pthread_mutex_lock(&Server::instance().sessionMutex); // session mutex lock

    /* notifyStartEANativeCb is called multiple times. limit this to one time with mEASessionStarted Flag */
    if (miAP2EASessionRunning == false)
    {
        miAP2EASessionRunning = true;

        LOG_INFO((tbdcl, "%s()  create Baidu session", __func__));
        Server::instance().startSession(wFile.c_str(), rFile.c_str());
    }

    pthread_mutex_unlock(&Server::instance().sessionMutex);   // session mutex unlock
}

void Server::onStopEANative(std::string wFile, std::string rFile)
{
    LOG_INFO((tbdcl, "Server::%s()  wFile:%s, rFile:%s", __func__, wFile.c_str(), rFile.c_str()));
    pthread_mutex_lock(&Server::instance().sessionMutex); // session mutex lock

     if (miAP2EASessionRunning == true)
     {
         miAP2EASessionRunning = false;

         if((Server::instance().sessionActive))
         {
             LOG_INFO((tbdcl, "%s()  destroy Baidu session", __func__));
             Server::instance().destroySession();
         }
         // stop and destroy EA Native session.
         stopEaSession();
         deinitEaSession();
     }

     pthread_mutex_unlock(&Server::instance().sessionMutex);   // session mutex unlock
}

bool Server::initEaSession(std::string inSerial)
{
    miAP2DeviceSerial = inSerial;
    miAP2Session = new iAP2EANativeSession(miAP2DeviceSerial,this);
    /* need to re-work on these */
    mGadgetLoader.setRollSwitchInfo(inSerial);
    mGadgetLoader.iap2FindVbus();

    return true;
}

bool Server::deinitEaSession()
{
    if (nullptr != miAP2Session) {
        delete miAP2Session;
        miAP2Session = nullptr;

        mGadgetLoader.switchHUToHost();
    }

    return true;
}

bool Server::startEaSession()
{
    int32_t res = 0;
    bool result = false;

    //switch Head-Unit to Gadget
    res = mGadgetLoader.switchHUToGadget();
    /* init iAP2Connection */
    if (0 == res) {
        res = miAP2Session->iAP2EASessionInit();
        LOG_INFO((tbdcl, "Server::%s()   iAP2EASessionInit()  = %d", __func__, res));
    }

    if (0 == res) {
        result =  true;
    }
    return result;
}

bool Server::stopEaSession()
{
    if (nullptr != miAP2Session) {
        miAP2Session->iAP2EASessionDeInit();
    }
    return true;
}

std::string to_string_server(DeviceInfo& inVal)
{
    std::stringstream ss;
    ss << "vendorId=0x" << std::hex << inVal.getiDeviceInfo(DSYS_IDVENDOR) <<
          " productId=0x" << std::hex << inVal.getiDeviceInfo(DSYS_IDPRODUCT) <<
          " serial=" << inVal.getDeviceInfo(DSYS_SERIAL);
    return ss.str();
}
} } // namespace adit { namespace bdcl {
